/*
 * Protection Domain
 *
 * Copyright (C) 2009-2011 Udo Steinberg <udo@hypervisor.org>
 * Economic rights: Technische Universitaet Dresden (Germany)
 *
 * This file is part of the NOVA microhypervisor.
 *
 * NOVA is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * NOVA is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License version 2 for more details.
 */

#pragma once

#include "compiler.h"
#include "crd.h"
#include "kobject.h"
#include "space_io.h"
#include "space_mem.h"
#include "space_obj.h"

class Pd : public Kobject, public Refcount, public Space_mem, public Space_io, public Space_obj
{
    private:
        static Slab_cache cache;

        WARN_UNUSED_RESULT
        mword clamp (mword,   mword &, mword, mword);

        WARN_UNUSED_RESULT
        mword clamp (mword &, mword &, mword, mword, mword);

    public:
        static Pd *current CPULOCAL_HOT;
        static Pd kern, root;

        INIT
        Pd (Pd *);

        Pd (Pd *own, mword sel, mword a) : Kobject (PD, own, sel, a) {}

        ALWAYS_INLINE HOT
        inline void make_current()
        {
            if (EXPECT_FALSE (htlb.chk (Cpu::id)))
                htlb.clr (Cpu::id);

            else if (EXPECT_TRUE (current == this))
                return;

            current = this;

            loc[Cpu::id].make_current();
        }

        ALWAYS_INLINE
        static inline Pd *remote (unsigned c)
        {
            return *reinterpret_cast<volatile typeof current *>(reinterpret_cast<mword>(&current) - CPULC_ADDR + CPUGL_ADDR + c * PAGE_SIZE);
        }

        ALWAYS_INLINE
        inline Mdb *lookup_crd (Crd crd)
        {
            switch (crd.type()) {
                case Crd::MEM:  return Space_mem::lookup_node (crd.base());
                case Crd::IO:   return Space_io::lookup_node (crd.base());
                case Crd::OBJ:  return Space_obj::lookup_node (crd.base());
            }

            return 0;
        }

        template <typename>
        void delegate (Pd *, mword, mword, mword, mword, mword = 0);

        template <typename>
        void revoke (mword, mword, mword, bool);

        void xfer_items (Pd *, Crd, Crd, Xfer *, Xfer *, unsigned long);

        void xlt_crd (Pd *, Crd, Crd &);
        void del_crd (Pd *, Crd, Crd &, mword = 0, mword = 0);
        void rev_crd (Crd, bool);

        ALWAYS_INLINE
        static inline void *operator new (size_t) { return cache.alloc(); }

        ALWAYS_INLINE
        static inline void operator delete (void *ptr) { cache.free (ptr); }
};
