/* SPDX-License-Identifier: MIT */
/*
 * (c) 2014 Maksym Planeta <mcsim.planeta@gmail.com>
 *          Matthias Lange <matthias.lange@kernkonzept.com>
 */

#include <iostream>
#include <unistd.h>

#include <l4/re/env>
#include <l4/re/error_helper>
#include <l4/vbus/vbus_gpio>
#include <l4/vbus/vbus>
#include <l4/cxx/exceptions>
#include <l4/cxx/ipc_timeout_queue>
#include <l4/re/util/object_registry>
#include <l4/cxx/unique_ptr>

using namespace std;

class My_loop_hooks :
  public L4::Ipc_svr::Timeout_queue_hooks<My_loop_hooks>,
  public L4::Ipc_svr::Ignore_errors
{
public:
  /**
   * method required for Timeout_queue
   */
  l4_kernel_clock_t now() { return l4_kip_clock(l4re_kip()); }
};

/**
 * This class implements the periodic state change of the LED
 */
class Blinking_led : public L4::Ipc_svr::Timeout_queue::Timeout
{
public:
  /**
   * interval has to be in microseconds
   */
  Blinking_led(L4::Ipc_svr::Server_iface *sif, L4vbus::Gpio_pin *led,
               unsigned interval)
    : _sif(sif), _led(led), _interval(interval)
  {
    // add ourselves to the server's timeout queue
    // absolute timeout with _interval length
    _sif->add_timeout(this, l4_kip_clock(l4re_kip()) + _interval);
  }

  /**
   * This function gets called when the timeout has expired
   */
  void expired()
  {
    unsigned v = _led->get();
    _led->set(~v & 0x1);
    _sif->add_timeout(this, timeout() + _interval);
  }

private:
  L4::Ipc_svr::Server_iface *_sif;
  L4vbus::Gpio_pin *_led;
  unsigned _interval;
};

L4Re::Util::Registry_server<My_loop_hooks> server;

int main()
{
  cout << "Hello, this is ex_gpio_led" << endl;

  try
    {
      L4::Cap<L4vbus::Vbus> vbus =
        L4Re::chkcap(L4Re::Env::env()->get_cap<L4vbus::Vbus>("vbus"),
                     "Could not find 'vbus' capability.\nCheck your lua config");

      L4vbus::Device root(vbus, L4VBUS_ROOT_BUS);
      L4vbus::Device dev;

      unsigned pin = ~0;
      l4vbus_device_handle_t gpio_handle = ~0;

      l4vbus_device_t dev_info;
      while (root.next_device(&dev, L4VBUS_MAX_DEPTH, &dev_info) == 0)
        {
          // find LED which is connected via GPIO
          if (dev.is_compatible("gpio-led") == 1)
            {
              // get device resources, e.g. GPIO pin number
              // this is needed to create the GPIO pin device later
              for (unsigned i = 0; i < dev_info.num_resources; ++i)
                {
                  l4vbus_resource_t res;
                  if (dev.get_resource(i, &res))
                    break;

                  if (res.type == L4VBUS_RESOURCE_GPIO)
                    {
                      pin = res.start;
                      gpio_handle = res.provider;
                    }
                }
              // we found a matching device, exit
              break;
            }
        }

      if (gpio_handle == ~0)
        throw L4::Runtime_error(-L4_ENODEV, "No compatible LED device found.");

      L4vbus::Device gpio_dev(vbus, gpio_handle);
      cxx::unique_ptr<L4vbus::Gpio_pin> led =
        cxx::make_unique<L4vbus::Gpio_pin>(gpio_dev, pin);

      // configure pin as Output
      L4Re::chksys(led->setup(L4VBUS_GPIO_SETUP_OUTPUT, 0),
                   "Failed to setup GPIO pin.");

      // 500ms interval
      Blinking_led b(&server, led.get(), 500000);

      led.release();

      server.loop();
    }
  catch (L4::Runtime_error &e)
    {
      cerr << "Runtime error: " << e.str() << ". Reason: " << e.extra_str()
           << endl;
    }

  return 0;
}
