/*
 * Copyright (C) 2009 Technische Universität Dresden.
 * Author(s): Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 *
 * License: see LICENSE.spdx (in this directory or the directories above)
 */

/**
 * \file
 * \brief  Support for the MPC52000
 */

#include "support.h"
#include "startup.h"
#include "platform.h"
#include <l4/drivers/uart_of.h>
#include <l4/drivers/of_if.h>
#include <l4/drivers/of_dev.h>

namespace {
class Platform_ppc_mpc52000 :
  public Platform_base,
  public Boot_modules_image_mode
{
  bool probe() override { return true; }
  Boot_modules *modules() override { return this; }

  void boot_kernel(unsigned long entry) override
  {
    typedef void (*func)(void *, unsigned long);
    L4_drivers::Of_if of_if;
    of_if.boot_finish();
    ((func)entry)(0, of_if.get_prom());
    exit(-100);
  }

  void init() override
  {
    // need to read stuff from Open Firmare device tree
    kuart.base_address = -1;
    kuart.base_baud = 115200;
    kuart.baud = 115200;
    kuart.irqno = -1;
    kuart.access_type  = L4_kernel_options::Uart_type_mmio;
    kuart_flags       |=   L4_kernel_options::F_uart_base
                         | L4_kernel_options::F_uart_baud
                         | L4_kernel_options::F_uart_irq;

    static L4::Uart_of _uart;
    static L4::Io_register_block_mmio r(kuart.base_address);
    _uart.startup(&r);
    set_stdio_uart(&_uart);
  }

  void setup_memory_map() override
  {
    L4_drivers::Of_if of_if;

    printf("  Detecting ram size ...\n");
    unsigned long ram_size = of_if.detect_ramsize();
    printf("    Total memory size is %lu MiB\n", ram_size / (1024 * 1024));
    mem_manager->ram->add(Region::start_size(0x0ULL, ram_size, ".ram",
                                             Region::Ram));

    // FIXME: move this somewhere else, it has nothing to do with memory setup
#if 0
    /* detect OF devices */
    unsigned long drives_addr, drives_length;

    if (of_if.detect_devices(&drives_addr, &drives_length))
      {
        mbi->flags |= L4UTIL_MB_DRIVE_INFO;
        mbi->drives_addr   = drives_addr;
        mbi->drives_length = drives_length;
      }
#endif
  }
};
}

REGISTER_PLATFORM(Platform_ppc_mpc52000);
