/*
 * Copyright (C) 2025 Adam Lackorzynski <adam@l4re.org>
 *
 * License: see LICENSE.spdx (in this directory or the directories above)
 */
#include <l4/re/util/object_registry>
#include <l4/re/util/br_manager>
#include <l4/re/rm>
#include <l4/re/remote_access>
#include <l4/re/util/cap_alloc>
#include <l4/sys/capability>
#include <l4/sys/debugger.h>
#include <l4/cxx/ipc_timeout_queue>

#include <l4/libunwind/backtrace_util.h>
#include <l4/re/dbg_events>

#include <stdio.h>

#include <list>

struct Backtrace_request
{
  L4::Cap<L4Re::Remote_access> ra;
  L4::Cap<L4Re::Rm> rm;
  l4_exc_regs_t regs;
};

static std::list<Backtrace_request> backtrace_request_list;

static L4Re::Util::Registry_server<L4Re::Util::Br_manager_timeout_hooks> server;

static int raif_access_remote_mem(l4_addr_t addr, l4_umword_t *valp, int write,
                                  void *arg)
{
  L4::Cap<L4Re::Remote_access> t(reinterpret_cast<l4_cap_idx_t>(arg));
  if (write)
    {
      printf("Write access not supported yet!\n");
      return -1;
    }

  l4_uint64_t v = 0;
  int r = t->read_mem(addr, L4Re::Remote_access::Wd_64bit, &v);
  if (r)
    return r;

  *valp = v;

  return 0;
}

class Backtrace_request_runner : public L4::Ipc_svr::Timeout_queue::Timeout
{
public:
  Backtrace_request_runner(L4::Ipc_svr::Server_iface *sif)
  : _sif(sif)
  {}

  void expired()
  {
    while (!backtrace_request_list.empty())
      {
        Backtrace_request btr = backtrace_request_list.front();
        backtrace_request_list.pop_front();

        libunwind_do_remote_backtrace(&btr.regs, btr.rm.cap(),
                                      &raif_access_remote_mem,
                                      reinterpret_cast<void *>(btr.ra.cap()));

        btr.ra->terminate(68);

        L4Re::Env::env()->task()->release_cap(btr.rm);
        L4Re::Util::cap_alloc.free(btr.rm);
        L4Re::Env::env()->task()->release_cap(btr.ra);
        L4Re::Util::cap_alloc.free(btr.ra);
      }
  }

private:
  L4::Ipc_svr::Server_iface *_sif;
};

static Backtrace_request_runner bt_req_runner(&server);

class Backtracer_server
: public L4::Epiface_t<Backtracer_server, L4Re::Dbg_events>
{
public:
  int op_request_backtrace(L4Re::Dbg_events::Rights,
                           l4_exc_regs_t regs,
                           L4::Ipc::Snd_fpage const &task,
                           L4::Ipc::Snd_fpage const &rmif);
};

int
Backtracer_server::op_request_backtrace(L4Re::Dbg_events::Rights,
                                        l4_exc_regs_t regs,
                                        L4::Ipc::Snd_fpage const &raif,
                                        L4::Ipc::Snd_fpage const &rmif)
{
  if (!raif.cap_received())
    return -L4_EINVAL;

  if (!rmif.cap_received())
    return -L4_EINVAL;

  int r;
  L4::Cap<L4Re::Remote_access> ra = server_iface()->template rcv_cap<L4Re::Remote_access>(0);
  r = server_iface()->realloc_rcv_cap(0);
  if (r < 0)
    return r;

  L4::Cap<L4Re::Rm> rm = server_iface()->template rcv_cap<L4Re::Rm>(1);
  r = server_iface()->realloc_rcv_cap(1);
  if (r < 0)
    return r;

  Backtrace_request btr;
  btr.ra = ra;
  btr.rm = rm;
  btr.regs = regs;
  backtrace_request_list.push_back(btr);

  server.add_timeout(&bt_req_runner, 1); // Immediately

  return 0;
}

int main(int argc, char **argv)
{
  (void)argc; (void)argv;

  Backtracer_server usrv;
  server.registry()->register_obj(&usrv, "backtracer");
  server.loop();

  return 0;
}
