#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <thread>
#include <chrono>
#include <unistd.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <cstring>
#include <sys/socket.h>

std::string get_hostname() {
    char hostname[256];
    if (gethostname(hostname, sizeof(hostname)) != 0) {
        return "Unknown";
    }
    return std::string(hostname);
}

void show_system_info() {
    struct utsname uts_info;
    if (uname(&uts_info) == 0) {
        std::cout << "=== System Information ===" << std::endl;
        std::cout << "Hostname: " << uts_info.nodename << std::endl;
        std::cout << "Kernel: " << uts_info.sysname << " " << uts_info.release << std::endl;
        std::cout << "Machine: " << uts_info.machine << std::endl;
    }
    
    std::cout << "\n=== Process Information ===" << std::endl;
    std::cout << "Current PID: " << getpid() << std::endl;
    std::cout << "Parent PID: " << getppid() << std::endl;
    
    DIR* dir = opendir("/proc");
    if (dir) {
        struct dirent* entry;
        int count = 0;
        while ((entry = readdir(dir)) != nullptr) {
            if (entry->d_type == DT_DIR) {
                bool is_number = true;
                for (char* p = entry->d_name; *p; p++) {
                    if (*p < '0' || *p > '9') {
                        is_number = false;
                        break;
                    }
                }
                if (is_number) {
                    count++;
                }
            }
        }
        closedir(dir);

        std::cout << "Processes visible in /proc: " << count << std::endl;
    } else {
        std::cout << "  Could not open /proc directory" << std::endl;
    }
}

void test_filesystem() {
    std::cout << "\n=== File System Tests ===" << std::endl;
    
    char cwd[256];
    if (getcwd(cwd, sizeof(cwd)) != nullptr) {
        std::cout << "Current directory: " << cwd << std::endl;
    }
    
    const char* paths[] = {"/", "/etc", "/tmp", "/proc"};
    for (const auto& path : paths) {
        struct stat info;
        if (stat(path, &info) == 0) {
            std::cout << "Can access: " << path << std::endl;
            
            DIR* dir = opendir(path);
            if (dir) {
                std::cout << "  Files in " << path << ": ";
                struct dirent* entry;
                int count = 0;
                while ((entry = readdir(dir)) != nullptr) {
                    std::cout << entry->d_name << " ";
                    count++;
                }
                std::cout << std::endl;
                closedir(dir);
            }
        } else {
            std::cout << "Cannot access: " << path << " (" << strerror(errno) << ")" << std::endl;
        }
    }
    
    std::cout << "Attempting to create file..." << std::endl;
    std::ofstream test_file("/tmp/test");
    if (test_file.is_open()) {
        test_file << "Hello from container!" << std::endl;
        test_file.close();
        std::cout << "Created file: /tmp/test" << std::endl;
    } else {
        std::cout << "Failed to create file: /tmp/test " << strerror(errno) << std::endl;
    }
}

void test_memory(int mb_to_allocate) {
    std::cout << "\n=== Memory Test ===" << std::endl;
    std::cout << "Trying to allocate " << mb_to_allocate << "MB of memory..." << std::endl;
    
    try {
        std::vector<char*> memory_chunks;

        for (int i = 0; i < mb_to_allocate; i++) {
            // Allocate 1MB
            const int chunk_size = 1024 * 1024;
            char* chunk = new char[chunk_size];
            
            for(int j = 0; j < chunk_size; j++) {
                chunk[j] = rand() % 0xFF;
            }
            
            memory_chunks.push_back(chunk);
            std::cout << "Allocated " << (i + 1) << "MB" << std::endl;
            
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        }
        
        std::cout << "Successfully allocated " << mb_to_allocate << "MB" << std::endl;
        
        for (auto chunk : memory_chunks) {
            delete[] chunk;
        }
    } catch (const std::bad_alloc& e) {
        std::cout << "Memory allocation failed: " << e.what() << std::endl;
        std::cout << "This may indicate cgroup memory limits are working" << std::endl;
    }
}

void test_cpu(int seconds) {
    std::cout << "\n=== CPU Test ===" << std::endl;
    std::cout << "Running CPU-intensive task for " << seconds << " seconds..." << std::endl;
    
    auto start_time = std::chrono::steady_clock::now();
    int duration_ms = seconds * 1000;
    
    while (true) {
        volatile double result = 0;
        for (int i = 0; i < 1000000; i++) {
            result += i * 1.1;
        }
        
        auto current_time = std::chrono::steady_clock::now();
        auto elapsed = std::chrono::duration_cast<std::chrono::milliseconds>(
            current_time - start_time).count();
            
        if (elapsed >= duration_ms) {
            break;
        }
    }
    
    std::cout << "CPU test completed" << std::endl;
}

void test_network() {
    std::cout << "\n=== Network Test ===" << std::endl;
    
    int sockfd = socket(AF_INET, SOCK_STREAM, 0);
    if (sockfd < 0) {
        std::cout << "Failed to create socket: " << strerror(errno) << std::endl;
    } else {
        std::cout << "Created socket successfully" << std::endl;
        close(sockfd);
    }
    
    std::cout << "Network interfaces:" << std::endl;
    std::ifstream net_dev("/proc/net/dev");
    if (net_dev.is_open()) {
        std::string line;
        std::getline(net_dev, line);
        std::getline(net_dev, line);
        
        while (std::getline(net_dev, line)) {
            size_t colon_pos = line.find(':');
            if (colon_pos != std::string::npos) {
                std::string iface = line.substr(0, colon_pos);
                iface.erase(0, iface.find_first_not_of(" \t"));
                std::cout << "  - " << iface << std::endl;
            }
        }
        net_dev.close();
    } else {
        std::cout << "  Could not read network interfaces" << std::endl;
    }
}

int main(int argc, char* argv[]) {
    srand(time(NULL));

    std::cout << "===== Container Isolation Test =====" << std::endl;
    std::cout << "This application tests various isolation features of the container" << std::endl;

    show_system_info();
    
    test_filesystem();
    
    test_network();
    
    int memory_to_allocate = 100;
    if (argc > 1) {
        memory_to_allocate = std::stoi(argv[1]);
    }
    
    int cpu_test_seconds = 5;
    if (argc > 2) {
        cpu_test_seconds = std::stoi(argv[2]);
    }
    
    test_memory(memory_to_allocate);
    test_cpu(cpu_test_seconds);
    
    std::cout << "\n===== Test Completed =====" << std::endl;
    return 0;
}