#include <iostream>
#include <string>
#include <vector>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sched.h>
#include <fstream>
#include <filesystem>
#include <cstring>
#include <cerrno>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <sys/types.h>
#include <iostream>

struct ContainerConfig {
    std::string name;
    std::string rootfs;
    std::string command;
    std::vector<std::string> args;
    
    int memory_limit_mb;
    int cpu_shares;
    int swap_limit_mb;
};

class CgroupManager {
public:
    bool setup(const std::string& cgroup_path) {
        this->base_path = cgroup_path;

        try {
            std::filesystem::create_directories(base_path);
        } catch(const std::filesystem::filesystem_error& e) {
            std::cerr << "Could not create cgroup directories for container: " << strerror(errno) << std::endl;
            return false;
        }

        return true;
    }

    bool cleanup() {
        try {
            std::filesystem::remove(base_path);
        } catch(const std::filesystem::filesystem_error& e) {
            std::cerr << "Could not cleanup cgroup directories for container: " << strerror(errno) << std::endl;
            return false;
        }

        return true;
    }

    bool set(const std::string& name, const std::string& value) const {
        std::string path = base_path + "/" + name;
        std::ofstream file(path);
        if (!file.is_open()) {
            std::cerr << "Failed to write to " << path << ": " << strerror(errno) << std::endl;
            return false;
        }
        file << value;
        return true;
    }

    bool set(const std::string& name, long value) const {
        return set(name, std::to_string(value));
    }

private:
    std::string base_path;
};


const std::string cgroup_base_path = "/sys/fs/cgroup/socker/";

const int CONTAINER_STACK_SIZE = 1024 * 1024;

CgroupManager cgroup;

bool enable_subtree_controllers(const std::vector<std::string>& controllers) {
    std::string subtree_control = cgroup_base_path + "/cgroup.subtree_control";
    std::ofstream out(subtree_control);
    if (!out.is_open()) {
        std::cerr << "Failed to open " << subtree_control << ": " << strerror(errno) << std::endl;
        return false;
    }

    for (const auto& ctrl : controllers) {
        out << "+" << ctrl << " ";
    }

    out << std::endl;
    return true;
}

int container_function(void* arg) {
    ContainerConfig& config = *static_cast<ContainerConfig*>(arg);
    
    /* TODO: move process into cgroup */
    /* HINT: `man cgroups` "procs" */

    /* TODO: overwrite the hostname as speicified in the ContainerConfig */
    /* HINT: `man 2 sethostname` */

    /* TODO: setup filesystem */
    /* HINT: `man 2 mount`, sysfs, proc, dev */
    /* HINT: `man 2 chroot`, `man 2 chdir` */

    std::vector<char*> args_ptrs;
    args_ptrs.push_back(const_cast<char*>(config.command.c_str()));
    for (const auto& arg : config.args) {
        args_ptrs.push_back(const_cast<char*>(arg.c_str()));
    }
    args_ptrs.push_back(nullptr);

    execvp(config.command.c_str(), args_ptrs.data());

    std::cerr << "execvp failed: " << strerror(errno) << std::endl;
    return 1;
}

int main(int argc, char* argv[]) {
    if (argc < 4) {
        std::cerr << "Usage: " << argv[0] << " <container_name> <rootfs_path> <command> [args...]" << std::endl;
        return 1;
    }

    ContainerConfig config;
    config.name = argv[1];
    config.rootfs = argv[2];
    config.command = argv[3];

    for (int i = 4; i < argc; i++) {
        config.args.push_back(argv[i]);
    }

    config.swap_limit_mb = 1; //disable swapping otherwise we cant observe the memory limit enforcement
    config.memory_limit_mb = 10;
    config.cpu_shares = 1024;
    
    if(!cgroup.setup(cgroup_base_path + config.name)) {
        return 1;
    }

    /* TODO: enable cgroups to set memory, cpu */
    /* HINT: `man cgroups` "subtree" */

    if(config.memory_limit_mb > 0) {
        /* TODO: enforce memory limit */
        /* HINT: https://www.kernel.org/doc/html/latest/admin-guide/cgroup-v2.html#memory-interface-files */
    }

    if(config.swap_limit_mb > 0) {
        /* TODO: enforce swap limit */
        /* HINT: https://www.kernel.org/doc/html/latest/admin-guide/cgroup-v2.html#memory-interface-files */
    }

    if(config.cpu_shares > 0) {
        /* TODO: enforce cpu shares limit */
        /* HINT: https://www.kernel.org/doc/html/latest/admin-guide/cgroup-v2.html#cpu-interface-files */
    }

    char* container_stack = (char*)malloc(CONTAINER_STACK_SIZE);

    /* TODO: start the container in a new namespace with according flags */
    /* HINT: `man clone` */
    pid_t container_pid = /* TODO */

    int status;
    waitpid(container_pid, &status, 0);

    cgroup.cleanup();

    /* TODO: cleanup the mounts */
    /* HINT: `man unmount` */
}